/* YACC/BISON parser */

%{
#include <stdio.h>
#include "node.h"
%}


/* list of needed tokens */

%token IDENTIFIER INT_DENOTATION
%token EQ_OP
%token AND_OP OR_OP
%token INT
%token IF ELSE WHILE CONTINUE BREAK RETURN

/* 
 * Surpress ambiguous shift/reduce conflict in compound statement.
 * This is predictably resolved by bison correctly anyway.       
 */

%expect 1

/* data types used in grammar */

%union
{
   int  sValue;
   char *pszValue;
   NODE *pNode;
}

/* rules which create nodes on the tree */

%type <pNode>   comp3100program
%type <pNode>   expression
%type <pNode>   condition
%type <pNode>   disjunction
%type <pNode>   conjunction
%type <pNode>   comparison
%type <pNode>   relation
%type <pNode>   sum
%type <pNode>   term
%type <pNode>   factor
%type <pNode>   primary
%type <pNode>   declaration_parameter_list
%type <pNode>   declaration_parameter
%type <pNode>   declaration_variable_list
%type <pNode>   declaration_variable
%type <pNode>   declaration_variable_init
%type <pNode>   statement
%type <pNode>   statement_list
%type <pNode>   statement_compound
%type <pNode>   statement_conditional
%type <pNode>   statement_iteration
%type <pNode>   statement_jump
%type <pNode>   statement_computation
%type <pNode>   identifier
%type <pNode>   int_denotation

/* nodes which return an integer type */

%type <sValue>   '='
%type <sValue>   INT_DENOTATION
%type <sValue>   '>'
%type <sValue>   '<'
%type <sValue>   EQ_OP

/* node which returns a string type */

%type <pszValue> IDENTIFIER

/* Evaluation precedence: lowest to highest */

%right '='
%left ','
%left OR_OP
%left AND_OP
%nonassoc '>' '<' EQ_OP
%left '+' '-'
%left '*' '/' '%'

/* the location of the root node */

%start comp3100program


/* Grammatical Rules for the comp3100 program syntax */

%%

/* root node */
comp3100program
	/* read an INT(eger) followed by an identifier then a left bracket followed by 
	 *declaration parameter list followed by right bracket then a statement compound
	 */

        : INT identifier '(' declaration_parameter_list ')' statement_compound
        {
            extern NODE* expTree;

	/* add this node to the tree */

            $$ = AddNode($4, NODE_PROGRAM, $6);

	/* set the root node to this current symbol */

            expTree = $$;
        } 
        ;

expression
        : identifier '=' expression
        {
                $$ = AddNode($1, NODE_ASSIGN, $3);
                ($$)->sTokenValue = $2;
        }
	/* if not an identifier the expression must then reduce to a condition  */
        | condition
        {
                $$ = $1;		/* set yyval to this value */
        }
        ;

condition
        : disjunction
        {
                $$ = $1;
        }
        | disjunction '?' expression ':' condition
        {
                $$ = AddNode($1, NODE_IF_ELSE, AddNode($3, NODE_ELSE, $5));
        }
        ;

disjunction
        : conjunction
        {
                $$ = $1;
        }
        | disjunction OR_OP conjunction
        {
                $$ = AddNode($1, NODE_OR_OP, $3);
        }
        ;

conjunction
        : comparison
        {
                $$ = $1;
        }
        | conjunction AND_OP comparison
        {
                $$ = AddNode($1, NODE_AND_OP, $3);
        }
        ;

comparison
        : relation 
        {
                $$ = $1;
        }
        | comparison EQ_OP relation
        {
                $$ = AddNode($1, NODE_EQ_OP, $3);
        }
        ;
        
relation
        : sum
        {
                $$ = $1;
        }
        | relation '<' sum
        {
                $$ = AddNode($1, NODE_LT_OP, $3);
        }
        | relation '>' sum
        {
                $$ = AddNode($1, NODE_GT_OP, $3);
        }
        ;

sum
        : term
        {
                $$ = $1;
        }
        | sum '+' term
        {
                $$ = AddNode($1, NODE_ADD_OP, $3);
        }
        | sum '-' term
        {
                $$ = AddNode($1, NODE_SUB_OP, $3);
        }
        ;

term
        : factor
        {
                $$ = $1;
        }
        | term '*' factor
        {
                $$ = AddNode($1, NODE_MUL_OP, $3);
        }
        | term '/' factor
        {
                $$ = AddNode($1, NODE_DIV_OP, $3);
        }
        | term '%' factor
        {
                $$ = AddNode($1, NODE_MOD_OP, $3);
        }
        ;

factor
        : primary
        {
                $$ = $1;
        }
        | '!' factor
        {
                $$ = AddNode($2, NODE_NOT_OP, NULL);
        }
        | '-' factor
        {
                $$ = AddNode($2, NODE_NEG_OP, NULL);
        }
        ;

primary
        : int_denotation
        {
                $$ = $1;
        }
        | identifier
        {
                $$ = $1;
        }
        | '(' expression ')'
        {
                /* $$ = AddNode($2, NODE_EXPRESSION, NULL); */
		$$ = $2;
        }
        ;

/*       DECLARATIONS AND STATEMENTS    */

declaration_parameter_list
        : declaration_parameter
        { /* add node */
              $$ = $1;
	}
        | declaration_parameter_list ',' declaration_parameter
	{
	$$ = $1;
            $$ = AppendNode($$, $3);	/* Append the node to the root tree  */
	}
        ;

declaration_parameter
        : INT identifier
        {
                ($2)->sVariableType = TYPE_INTEGER;
                $$ = AddNode($2, NODE_DECLARATION, NULL);
        }
        ;

declaration_variable_list
        : declaration_variable
        {
                $$ = $1;
        }
        | declaration_variable_list declaration_variable
	{
		$$ = $1;
            $$ = AppendNode($$, $2);
	}
        ;

declaration_variable
        : INT declaration_variable_init
	{
		$$ = $2;
	}
        ;

declaration_variable_init
        : identifier ';'
        {
              $$ = AddNode($1, NODE_DECLARATION, NULL);
        }
        | identifier '=' expression ';'
        {
            $$ = AddNode($1, NODE_DECLARATION, $3);
            ($$)->sVariableType = TYPE_INTEGER;
        }
        | identifier '=' expression ',' declaration_variable_init
        {
            $$ = AddNode($1, NODE_DECLARATION, $3);
            ($$)->sVariableType = TYPE_INTEGER;
            $$ = AppendNode($$, $5);
        }
        | identifier ',' declaration_variable_init
        {
            $$ = AddNode($1, NODE_DECLARATION, NULL);
            $$ = AppendNode($$, $3);
        }
        ;

statement
        : statement_compound
        {
                $$ = $1;
        }
        | statement_conditional
        {
                $$ = $1;
        }
        | statement_iteration
        {
                $$ = $1;
        }
        | statement_jump
        {
                $$ = $1;
        }
        | statement_computation
        {
                 $$ = $1;
        }
        ;

statement_list
        : statement
        {
		$$ = $1;
        }
        | statement_list statement
        {
		$$ = $1;
            	$$ = AppendNode($$, $2);
        }
        ;

statement_compound
        : '{' '}'
        {
                $$ = 0;
        }
        | '{' statement_list '}'
        {
                $$ = AddNode(NULL, NODE_BLOCK, $2)
        }
        | '{' declaration_variable_list '}' 
        {
                $$ = AddNode($2, NODE_BLOCK, NULL);
        }
        | '{' declaration_variable_list statement_list '}'
	{
 		$$ = AddNode($2, NODE_BLOCK, $3);
	}
        ;

/*
 * Contains a dangling else ambiguity. Bison chooses shift instead of reduce
 * therefore attaches the else clause to the inner-most if-statement.      
 */

statement_conditional
        : IF '(' expression ')' statement
        {
                $$ = AddNode($3, NODE_IF, $5);
        }
        | IF '(' expression ')' statement ELSE statement
        {
                $$ = AddNode($3, NODE_IF_ELSE, AddNode($5, NODE_ELSE, $7));
        }
        ;

statement_iteration
        : WHILE '(' expression ')' statement
        {
                $$ = AddNode($3, NODE_WHILE, $5);
        }
        ;

statement_jump
        : BREAK ';'
        {
                $$ = AddNode(NULL, NODE_BREAK, NULL);
        }
        | CONTINUE ';'
        {
                $$ = AddNode(NULL, NODE_CONTINUE, NULL);
        }
        | RETURN expression ';'
        {
                $$ = AddNode($2, NODE_RETURN, NULL);
        }
        ;

statement_computation
        : ';'
        {
                $$ = 0;
        }
        | expression ';'
        {
		$$ = $1;
        }
        ;

identifier
        : IDENTIFIER
        {
                $$ = AddVariable($1);	/* create a node with the value equal to this identifier */
        }
        ;

int_denotation
        : INT_DENOTATION
        {
                $$ = AddNumber($1);	/* create a ndoe with the value equal to this digit */
        }
        ;

%% 	/* end of rules */
