/* Chatter Server
 *
 * Authors:
 *  Benjamin Ranck       bran7670@mail.usyd.edu.au
 *  Manpreet Lehl        manpreet@student.usyd.edu.au
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/select.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/types.h>

#include <netinet/in.h>
#include <arpa/inet.h>
#include <unistd.h>

#include "schatter.h"
#include "schatter_x.h"
#include "client.h"
#include "configure.h"
#include "messages.h"


/* This file descriptor set is our master set and contains the file descriptor
 * of the listening socket and the file descriptor of all clients which are
 * connected to the server. */
fd_set socks;

/* This contains the number of file descriptors in our socks fd_set. */
int fdmax;

/* This contains the file descriptor of the listening socket. */
int socklisten;

/* This array holds the details of every ciient connected to the server. */
struct client clients[MAX_CLIENTS];


int main(int argc, char** argv) {
    fd_set readfds; /* Select modifies this set. */
	struct sockaddr_in servaddr; /* use internet socket struct for bind */
	int reuse_addr=1;

	if ((socklisten = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		puts("ERROR main(int,char**): Listening socket could not be created.");
		exit(EXIT_FAILURE);  /* Exit indicating an error. */
	}

	/* So that we can re-bind to it without TIME_WAIT problems.
	 * Taken from UNIX socket FAQ. */
	if ( (setsockopt(socklisten, SOL_SOCKET, SO_REUSEADDR, &reuse_addr,
    		sizeof(reuse_addr)) )< 0) {
		puts("ERROR main(int,char**): Socket option could not be set.");
		exit(EXIT_FAILURE);  /* Exit indicating an error. */
	}

	/*bind */
	servaddr.sin_family=AF_INET;
	servaddr.sin_addr.s_addr = htonl(INADDR_ANY);
	servaddr.sin_port = htons(SERVER_PORT);
	memset(&(servaddr.sin_zero), '\0', 8);  /*set the rest to zero */

	if( (bind(socklisten, (struct sockaddr *)&servaddr,
	sizeof(servaddr) )) < 0) {
		printf("ERROR main(int,char**): Could not bind socket to listen port: "
            "port %d.", SERVER_PORT);
		exit(EXIT_FAILURE); /* Exit indicating an error. */
	}
	/*listen*/
	if (listen(socklisten, LISTEN_QUEUE_SIZE) == -1) {	/* this listens using
                                                         * our socklisten */
		puts("ERROR main(int,char**):  Could not listen.");
		exit(EXIT_FAILURE); /* Exit indicating an error. */
}

	memset(clients, '\0', sizeof(clients) );
	FD_ZERO(&socks);	/*clears out the list */
	FD_ZERO(&readfds);
	FD_SET(socklisten, &socks); /*add the listening socket to the socks list */

	fdmax = socklisten;

	for(;;) { /* endless loop */
		/*try select */
		readfds = socks;
		if( (select(fdmax+1, &readfds, NULL, NULL, NULL)) < 0) { /*keep going until we select something*/
			puts("ERROR main(int,char**): With select.");
			exit(EXIT_FAILURE); /* Exit indicating there was an error. */
		}
		/*now run through all the fd's to see if anything is happening. */
		readSocks(readfds);
	}

    /* Exit normally. */
	return EXIT_SUCCESS;
}

void addNewSock(int newsock) {
	int newfd, i;
	int clilen;
	struct sockaddr_in cliaddr;
	clilen = sizeof(cliaddr);
	if( (newfd = accept(newsock, (struct sockaddr*)& cliaddr, (unsigned int*) &clilen)) < 0 ) {
		puts("ERROR addNewSock(int): Could not accept new connection.");
	}
	else {
		/* connection accepted so try and find spot for new client */
		for(i=0; (i < MAX_CLIENTS) && (newfd != -1) ; ++i) {
			if(clients[i].intFD == 0) {
				clients[i].intFD = newfd;
                client_execConnect(i, inet_ntoa(cliaddr.sin_addr));
				FD_SET(newfd, &socks);
				if(fdmax < newfd)
					fdmax = newfd;
				newfd = -1; /*stop once we've found a slot */
			}
		}/*end maxclient for loop */
		if(newfd != -1) { /*means no slot was found */
			printf("SERVER Connection rejected (server is full):  IP=%s\n",
				inet_ntoa(cliaddr.sin_addr));
			send(newfd, MSG_SERVER_FULL, sizeof(MSG_SERVER_FULL), 0);
			/*don't really care if send is unsuccessful*/
			close(newfd); /* Close the rejected connection. */
		}
	}/* end else */
}


void handleClient(int clientsock) {
	int bufsize;
	char buf[MSG_LENGTH];

	memset(buf, '\0', sizeof(buf) );

    /* test if client there */
    /* The call to recv(...) has been changed to read(...) as recv can cause
     * bugger over flows when there is more data available then what can fit
     * into our buf variable. */
	if( (bufsize = read(clients[clientsock].intFD, buf, sizeof(buf))) <= 0) {
		if(bufsize < 0) {
			printf("ERROR handleClient(int): In recv for client:  "
                "FD=%d  Slot=%d\n", clients[clientsock].intFD, clientsock);
		}
		if(bufsize == 0) {
            /* This means the client has disconnected. */
		}
        client_execDisconnect(clientsock);
	}
	else {
        /* Set the last character of our buf to be a null terminator. */
        buf[MSG_LENGTH - 1] = '\0';
        if (buf[0] == '/') {
            /* The data is a server command. */
            handleCommand(clientsock, buf, bufsize);
        }
        else {
            /* The data is a normal message. */
            handleMessage(clientsock, buf, bufsize);
        }
	}
}


/*Loops through all the sockets checking for activity */
void readSocks(fd_set readfds) {
	int i;

	if(FD_ISSET(socklisten, &readfds)) {/* Means we have a new connection */
				addNewSock(socklisten);
	}

	for(i=0; i<MAX_CLIENTS; ++i) {
        /* Check if any activity occured on this connection. */
		if(FD_ISSET(clients[i].intFD, &readfds) ) {
            DEBUG printf("DEBUG readSocks(): Handling client:  "
                "FD=%d  Slot=%d  IP=%s\n",
                clients[i].intFD, i, clients[i].strIP);
			handleClient(i);
		}
	}

}
