/* Chatter Server Extensions
 *
 * Authors:
 *  Benjamin Ranck       bran7670@mail.usyd.edu.au
 *  Manpreet Lehl        manpreet@student.usyd.edu.au
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "schatter.h"
#include "schatter_x.h"
#include "client.h"
#include "configure.h"
#include "messages.h"
#include "util.h"


void handleCommand (int slot, char* msg, size_t msglen) {

    /* An array of 5 strings. This equals one command and a maximum of 2
     * parameters. */
    char tokens[3][MSG_LENGTH];

    /* The number of tokens we extracted from msg. */
    int intTokens = 0;

    /* A pointer to a string which we will use to test what strtok() returns. */
    char* strptr = NULL;


    /* Set tokens to NULL. */
	memset(tokens, '\0', sizeof(tokens));

    DEBUG printf("DEBUG handleCommand(int,char*,int): "
        "Received command:  FD=%d Slot=%d IP=%s\n",
        clients[slot].intFD, slot, clients[slot].strIP);

    /* Since msg may contain trailing "\r\n" characters we will use strtok()
     * and the delimeter "\r\n" to remove this character, ie get the first
     * token. */
    strptr = strtok(msg, "\r\n");
    if (strptr != NULL) {
        /* This means their was "\r\n" characters. When we copy our msg to
         * command we also remove the leading '/' character as well for
         * convenience. */
        strcpy(msg, strptr+1);
    }
    else {
        /* This means their was no '\n' character. This is just for safety as
         * usually the client sends a line at a time and this includes the
         * trailing '\n' character. Nothing needs to be done as msg doesn't have
         * a trailing '\n'. */
    }

    /* Get the first token from msg. This is the server command (including the
     * leading slash ('/'), eg: '/name'. */
    strptr = strtok(msg, " ");
    if (strptr != NULL) {
        strcpy(tokens[0], strptr);
        intTokens++;

        /* Get the second token from the msg. This is the first parameter. */
        strptr = strtok(NULL , " ");
        if (strptr != NULL) {
            strcpy(tokens[1], strptr);
            intTokens++;

            /* Get the third token from the msg. This is the second parameter.*/
            strptr = strtok(NULL, " ");
            if (strptr != NULL) {
                strcpy(tokens[2], strptr);
                intTokens++;

            }
        }
    }
    else {
        /* This means strptr was NULL and occurs when no command was specified
         * after the '/' character. In this situation we just simply ignore the
         * whole command altogether. */
    }

    if (intTokens == 0) {
        /* No command was specified so we just pretend the client didn't send
         * anything and reply with the command list help. */
        sendMessage(slot, MSG_HELP_COMMANDS, sizeof(MSG_HELP_COMMANDS));
    }
    else if (strcmp("details", tokens[0]) == 0) {
        /* Client sent the details command:
         * /details */
        if (intTokens == 1) {
            /* Execute /details command. */
            client_execDetails(slot);
        }
        else {
            /* Reply with the command help. */
            sendMessage(slot, MSG_HELP_DETAILS, sizeof(MSG_HELP_DETAILS));
        }
    }
    else if (strcmp("channel", tokens[0]) == 0) {
        /* Client sent the channel command:
         * /channel <channel> */
        if (intTokens == 2) {
            /* Execute /channel command */
            client_execChannel(slot, tokens[1]);
        }
        else {
            /* Reply with the command help. */
            sendMessage(slot, MSG_HELP_CHANNEL, sizeof(MSG_HELP_CHANNEL));
        }
    }
    else if (strcmp("name", tokens[0]) == 0) {
        /* Client sent the name command:
         * /name <name> */
        if (intTokens == 2) {
            /* Execute /name command */
            client_execName(slot, tokens[1]);
        }
        else {
            /* Reply with the command help. */
            sendMessage(slot, MSG_HELP_NAME, sizeof(MSG_HELP_NAME));
        }
    }
    else if (strcmp("users", tokens[0]) == 0) {
        /* Client sent the users command:
         * /users */
        if (intTokens == 1) {
            /* Execute /users command */
            client_execUsers(slot);
        }
        else {
            /* Reply with the command help. */
            sendMessage(slot, MSG_HELP_USERS, sizeof(MSG_HELP_USERS));
        }
    }
    else if (strcmp("list", tokens[0]) == 0) {
        /* Client sent the users command:
         * /users */
        if (intTokens == 1) {
            /* Execute /users command */
            client_execList(slot);
        }
        else {
            /* Reply with the command help. */
            sendMessage(slot, MSG_HELP_USERS, sizeof(MSG_HELP_USERS));
        }
    }
    else if (strcmp("quit", tokens[0]) == 0) {
        /* Client sent the users command:
         * /users */
        if (intTokens == 1) {
            /* Execute /users command */
            client_execDisconnect(slot);
        }
        else {
            /* Reply with the command help. */
            sendMessage(slot, MSG_HELP_QUIT, sizeof(MSG_HELP_QUIT));
        }
    }
    else {
        /* The command was not recognised so display a simple help message. */
        sendMessage(slot, MSG_CMD_UNKNOWN, sizeof(MSG_CMD_UNKNOWN));
    }
}


void handleMessage (int slot, char* msg, size_t msglen) {

    char* strptr = NULL;

    /* This needs to be longer then MSG_LENGTH as adds information to the
     * original message. */
    char newmsg[2 * MSG_LENGTH];

    memset(newmsg, '\0', sizeof(newmsg));

    /* Just sends the message to every client on the same channel as the client
     * who sent the message except to the client which sent it. */

    DEBUG printf("DEBUG handleMessage(int,char*,int): "
        "Received message:  FD=%d Slot=%d IP=%s\n",
        clients[slot].intFD, slot, clients[slot].strIP);

    /* Since msg may contain trailing "\r\n" characters we will use strtok()
     * and the delimeter "\r\n" to remove this character, ie get the first
     * token. */
    strptr = strtok(msg, "\r\n");
    if (strptr != NULL) {
        /* This means their was "\r\n" characters. */
        Strncpy(msg, msg, msglen);
    }
    else {
        /* This means their was no '\n' character. This is just for safety as
         * usually the client sends a line at a time and this includes the
         * trailing '\n' character. Nothing needs to be done as msg doesn't have
         * a trailing '\n'. */
    }

    sprintf(newmsg, "%s: %s\n", clients[slot].strName, msg);

    sendChannel(slot, clients[slot].strChannel, newmsg, sizeof(newmsg));

}

/* Send the given message to the specified client. */
void sendMessage (int slot, char* msg, size_t msglen) {

   send(clients[slot].intFD, msg, msglen, 0);

}

/* Send the given message to all users on the specified channel, except the user
 * with the given client id (slot). */
void sendChannel (int slot, char* channel, char* msg, size_t msglen) {

    int i;

    /* Loop through every user. */
    for (i=0; i < MAX_CLIENTS; ++i) {

        /* Now check if they are on the specified channel and that they aren't
         * the user who is exempt. */
        if ((strcmp(channel, clients[i].strChannel) == 0) && (i != slot)) {

            /* Now send the message to this client. */
            sendMessage(i, msg, msglen);

        }
    }
}
